#!/usr/bin/env python3
import argparse, json, os, sys, glob

GATE = {
    "band_cv_max": 0.10,
    "gap_rel_max": 0.10,
    "seg_r2_min": 0.80,
    "seg_vis_min": 0.20,
    "inner_az_flat_max": 0.03
}

def latest_run_dir(base="runs"):
    runs = sorted(glob.glob(os.path.join(base, "*")))
    return runs[-1] if runs else None

def load_audit(run_dir):
    p = os.path.join(run_dir, "outputs", "audits", "h21_audit.json")
    with open(p, "r", encoding="utf-8") as f:
        return json.load(f)

def load_result_line(run_dir):
    p = os.path.join(run_dir, "outputs", "run_info", "result_line.txt")
    return open(p, "r", encoding="utf-8").read().strip()

def main():
    ap = argparse.ArgumentParser(description="Verify H21 PASS and print metrics")
    ap.add_argument("--run_dir", default=None, help="Path to a run dir under ./runs")
    args = ap.parse_args()

    run_dir = args.run_dir or latest_run_dir()
    if not run_dir:
        print("No runs found. Run: python3 scripts/run_h21.py", file=sys.stderr)
        sys.exit(2)

    aud = load_audit(run_dir)
    line = load_result_line(run_dir)

    ok = bool(aud.get("pass") is True)
    print("result_line:", line)
    print("PASS flag:", ok)
    if not ok:
        print(json.dumps(aud, indent=2))
        sys.exit(1)

    # Sanity checks vs gates
    bm = aud["band_metrics"]
    inner_cv = bm[0]["cv"]
    outer_cv = bm[1]["cv"]
    gr = aud["gap_ratio"]
    r2 = aud["seg_metrics"]["r2K"]
    vis = aud["seg_metrics"]["visibility"]
    az = aud["az_flat_inner"]

    checks = [
        ("inner_cv <= band_cv_max", inner_cv <= GATE["band_cv_max"]),
        ("outer_cv <= band_cv_max", outer_cv <= GATE["band_cv_max"]),
        ("gap_ratio <= gap_rel_max", gr <= GATE["gap_rel_max"]),
        ("seg_r2 >= seg_r2_min", r2 >= GATE["seg_r2_min"]),
        ("seg_vis >= seg_vis_min", vis >= GATE["seg_vis_min"]),
        ("az_flat_inner <= inner_az_flat_max", az <= GATE["inner_az_flat_max"]),
    ]

    for name, passed in checks:
        print(f"{name}: {passed}")
    if not all(p for _, p in checks):
        sys.exit(1)

    print("Verification: PASS ✅")

if __name__ == "__main__":
    main()
